<?php

namespace App\Models;

use App\Constants\OrderConstant;
use App\Constants\StatusCode;
use App\Helpers\Functions;
use App\User;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    protected $guarded = ['id'];

    public static $statusAll = [
        OrderConstant::STATUS_DON_LUU,
        OrderConstant::STATUS_CHOT_DON,
        OrderConstant::STATUS_HUY_DON,
        OrderConstant::STATUS_CHO_XAC_NHAN,
        OrderConstant::STATUS_HOAN_HUY,
        OrderConstant::STATUS_DA_DANG,
        OrderConstant::STATUS_GIAO_NGAY,
        OrderConstant::STATUS_HOAN_GIAO_HANG,
        OrderConstant::STATUS_DA_LAY_HANG,
        OrderConstant::STATUS_KHONG_LAY_DUOC_HANG,
        OrderConstant::STATUS_DANG_GIAO_HANG,
        OrderConstant::STATUS_DA_GIAO_HANG,
        OrderConstant::STATUS_DA_THANH_TOAN,
        OrderConstant::STATUS_KHONG_GIAO_DUOC,
        OrderConstant::STATUS_YEU_CAU_GIAO_LAI,
        OrderConstant::STATUS_DANG_HOAN,
    ];

    public static $statusDelivery = [
        OrderConstant::STATUS_DA_DANG,
        OrderConstant::STATUS_DA_LAY_HANG,
        OrderConstant::STATUS_GIAO_NGAY,
        OrderConstant::STATUS_KHONG_LAY_DUOC_HANG,
        OrderConstant::STATUS_YEU_CAU_GIAO_LAI,
    ];

    public static $statusHoan = [
        OrderConstant::STATUS_HOAN_HUY,
        OrderConstant::STATUS_DANG_HOAN,
    ];

    public function orderDetails()
    {
        return $this->hasMany(OrderDetail::class);
    }

    public function get_district_ward()
    {
        $location['district'] = Location::where('code', $this->district_id)->first();
        $location['ward'] = Location::where('code', $this->ward_id)->first();
        return $location;
    }

    public function getProductTextAttribute()
    {
        $array_service = OrderDetail::select('product_id')->where('order_id', $this[0]->id)->pluck('product_id')->toArray();
        $products = Product::whereIn('id', $array_service)->get();
        return implode(',</br> ', $products);
    }

    public function getProductSpaceAttribute()
    {
        $array_service = OrderDetail::select('product_id')->where('order_id', $this->id)->pluck('product_id')->toArray();
        $products = Product::whereIn('id', $array_service)->pluck('name')->toArray();
        return implode(' ;', $products);
    }

    public function getProductCodeAttribute()
    {
        $result = [];
        $array_service = OrderDetail::select('product_id', 'quantity')->where('order_id', $this->id)->get();
        if (count($array_service)) {
            foreach ($array_service as $item) {
                $products = Product::find($item->product_id);
                if (!empty($products)) $result[] = '(' . $item->quantity . ')' . $products->code;
            }
        }
        return implode(' ;', $result);
    }

    public function getQuantityTextAttribute()
    {
        $array_service = OrderDetail::select('quantity')->where('order_id', $this->id)->pluck('quantity')->toArray();
        return implode(';', $array_service);
    }

    public function orderStatus()
    {
        return $this->belongsTo(OrderStatus::class, 'status', 'id');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customer_id', 'id');
    }

    public function sale()
    {
        return $this->belongsTo(User::class, 'sale_id', 'id');
    }

    public function warehouse()
    {
        return $this->belongsTo(User::class, 'warehouse_id', 'id');
    }

    public function marketing()
    {
        return $this->belongsTo(User::class, 'mkt_id', 'id');
    }

    public function depot()
    {
        return $this->belongsTo(Depot::class);
    }

    public function getOrderTextAttribute()
    {
        $array_service = OrderDetail::where('order_id', $this->id)->orderBy('product_id', 'asc');

        $all_product_id = $array_service->pluck('product_id')->toArray();
        $product_text = Product::whereIn('id', $all_product_id)->pluck('name')->toArray();
        $data['product_text'] = implode('</br> ', $product_text);

        $product_quantity = $array_service->pluck('quantity')->toArray();
        $data['product_quantity'] = implode('</br> ', $product_quantity);

        $product_price = $array_service->get()->map(function ($m) {
            return $m = number_format($m->price);
        });
        $product_total_price = $array_service->get()->map(function ($m) {
            return $m = number_format($m->total_price);
        });

        $data['product_price'] = implode('</br> ', $product_price->toArray());
        $data['product_total_price'] = implode('</br> ', $product_total_price->toArray());
        return $data;
    }

    public static function search($input, $orderBy = "id", $sort = 'ngay_chot_don')
    {
        $docs = self::when(isset($input['start_date']) && isset($input['end_date']), function ($q) use ($input, $sort) {
//            if (isset($input['role_filter']) && $input['role_filter'] == 'mkt') {
            $q->whereBetween($sort, [
                Functions::yearMonthDayTime($input['start_date']),
                Functions::yearMonthDayTime($input['end_date']),
            ]);

        })->when(isset($input['all_sale_id']), function ($q) use ($input) {
            $q->whereIn('sale_id', $input['all_sale_id']);
        })->when(isset($input['arr_mkt_id']), function ($q) use ($input) {
            $q->whereIn('mkt_id', $input['arr_mkt_id']);
        })->when(isset($input['sale_id']), function ($q) use ($input) {
            $q->where('sale_id', $input['sale_id']);
        })->when(isset($input['marketing']), function ($q) use ($input) {
            $q->where('mkt_id', $input['marketing']);
        })->when(isset($input['status']), function ($q) use ($input) {
            $q->where('status', $input['status']);
        })->when(isset($input['doi_soat']), function ($q) use ($input) {
            $q->where('doi_soat', $input['doi_soat']);
        })->when(isset($input['warehouse_id']), function ($q) use ($input) {
            $q->where('warehouse_id', $input['warehouse_id']);
        })->when(isset($input['orderDepot']), function ($q) use ($input) {
            $q->where('status', '>', 0);
        })->when(isset($input['orderDone']), function ($q) use ($input) {
            $q->where('status', '>', OrderConstant::STATUS_CHOT_DON)->where('status', '<>', OrderConstant::STATUS_HUY_DON);
        })->when(isset($input['not_in_status']), function ($q) use ($input) {
            $q->whereNotIn('status', $input['not_in_status']);
        })->when(isset($input['in_status']), function ($q) use ($input) {
            $q->whereIn('status', $input['in_status']);
        })->when(isset($input['all_sale_id']), function ($q) use ($input) {
            $q->whereIn('sale_id', $input['all_sale_id']);
        })->when(isset($input['arr_mkt_id']), function ($q) use ($input) {
            $q->whereIn('mkt_id', $input['arr_mkt_id']);
        })->when(isset($input['arr_ids']), function ($q) use ($input) {
            $q->whereIn('id', $input['arr_ids']);
        })->when(isset($input['customer_id']), function ($q) use ($input) {
            $q->whereIn('customer_id', $input['customer_id']);
        })->when(isset($input['depot_id']), function ($q) use ($input) {
            $q->where('depot_id', $input['depot_id']);
        })->when(isset($input['order_status']), function ($q) use ($input) {
            $q->where('status', $input['order_status']);
        })->when(isset($input['hoa_hong']), function ($q) use ($input) {
            $q->where('hoa_hong', $input['hoa_hong']);
        })->when(isset($input['goi_dau']), function ($q) use ($input) {
            $q->where('goi_dau', $input['goi_dau']);
        })->when(isset($input['phone']), function ($q) use ($input) {
            if (is_numeric($input['phone'])) {
                $customers = Customer::select('id')->where('phone', $input['phone'])->pluck('id')->toArray();
                $q->whereIn('customer_id', $customers);
            } else {
                $q->where('code', $input['phone']);
            }

        })->when(isset($input['product_id']), function ($q) use ($input) {
            $q->whereHas('orderDetails', function ($qw) use ($input) {
                $qw->where('product_id', $input['product_id']);
            });
        })->when(isset($input['arr_product_id']), function ($q) use ($input) {
            $q->whereHas('orderDetails', function ($qw) use ($input) {
                $qw->whereIn('product_id', $input['arr_product_id']);
            });
        })->when(isset($input['customer_old']), function ($q) use ($input) {
            $q->whereHas('customer', function ($qw) use ($input) {
                $qw->where('customer_old', $input['customer_old']);
            });
        })->when(isset($input['is_upsale']), function ($q) use ($input) {
            $q->where('is_upsale', $input['is_upsale']);
        })->when(isset($input['gioi_thieu']), function ($q) use ($input) {
            $q->whereHas('customer', function ($qw) use ($input) {
                $qw->where('gioi_thieu', $input['gioi_thieu']);
            });
        })->when(isset($input['source_id']), function ($q) use ($input) {
            $q->whereHas('customer', function ($qw) use ($input) {
                $qw->where('source_id', $input['source_id']);
            });
        })
            ->orderByDesc($orderBy);
        return $docs;
    }

    public static function revenueWithStatus($input, $orderBy = "id")
    {
        $data = self::select('mkt_id', 'sale_id', \DB::raw('count(id) as count'), \DB::raw('SUM(total) as sum'), \DB::raw('SUM(advance_price) as sum_advance_price'), \DB::raw('SUM(ship_price) as sum_ship_price'))
            ->when(isset($input['start_date']) && isset($input['end_date']), function ($q) use ($input) {
                $q->whereBetween('ngay_chot_don', [
                    Functions::yearMonthDayTime($input['start_date']),
                    Functions::yearMonthDayTime($input['end_date']),
                ]);
            })->when(isset($input['mkt_arr']), function ($q) use ($input) {
                $q->whereIn('mkt_id', $input['mkt_arr']);
            })->when(isset($input['sale_arr']), function ($q) use ($input) {
                $q->whereIn('sale_id', $input['sale_arr']);
            })->when(isset($input['doi_soat']), function ($q) use ($input) {
                $q->where('doi_soat', $input['doi_soat']);
            })->when(isset($input['status_arr']), function ($q) use ($input) {
                $q->whereIn('status', $input['status_arr']);
            })->when(isset($input['product_id']), function ($q) use ($input) {
                $orderDetails = OrderDetail::where('product_id', $input['product_id'])->pluck('order_id')->toArray();
                $q->whereIn('id', $orderDetails);
            })->when(isset($input['gioi_thieu']), function ($q) use ($input) {
                $q->whereHas('customer', function ($qw) use ($input) {
                    $qw->where('gioi_thieu', $input['gioi_thieu']);
                });
            })
            ->groupBy($orderBy)->with('sale', 'marketing');
        return $data;
    }

    public static function searchStatus($status, $query)
    {
        $query = $query->whereIn('status', $status);
        return $query;
    }

    public static function searchGroupBy($input, $orderBy = "id", $groupBy = "sale_id")
    {
        $docs = self::select('sale_id', 'mkt_id', \DB::raw('SUM(total) as all_total'), \DB::raw('SUM(advance_price) as all_advance_price'), \DB::raw('SUM(ship_price) as all_ship_price'))->groupBy($groupBy)
            ->when(isset($input['start_date']) && isset($input['end_date']), function ($q) use ($input) {
                $q->whereBetween('ngay_chot_don', [
                    Functions::yearMonthDayTime($input['start_date']),
                    Functions::yearMonthDayTime($input['end_date']),
                ]);
            })->when(isset($input['sale_id']), function ($q) use ($input) {
                $q->where('sale_id', $input['sale_id']);
            })->when(isset($input['mkt_id']), function ($q) use ($input) {
                $q->where('mkt_id', $input['mkt_id']);
            })->when(isset($input['status']), function ($q) use ($input) {
                $q->where('status', $input['status']);
            })->when(isset($input['not_in_status']), function ($q) use ($input) {
                $q->whereNotIn('status', $input['not_in_status']);
            })->when(isset($input['all_sale_id']), function ($q) use ($input) {
                $q->whereIn('sale_id', $input['all_sale_id']);
            })->when(isset($input['is_upsale']) && $input['is_upsale'], function ($q) use ($input) {
                $q->where('is_upsale', $input['is_upsale'])->whereHas('customer', function ($qw) {
                    $qw->where('gioi_thieu', StatusCode::INACTIVE);
                });
            })->orderByDesc($orderBy);
        return $docs;
    }

    public function productDepot()
    {
        $product_depot = ProductDepot::where('depot_id', $this->depot_id)->with('product')->get();
        return $product_depot;
    }

    public function city()
    {
        return $this->belongsTo(Location::class, 'city_id', 'code');
    }

    public function district()
    {
        return $this->belongsTo(Location::class, 'district_id', 'code');
    }

    public function ward()
    {
        return $this->belongsTo(Location::class, 'ward_id', 'code');
    }

    public function getAddressTextAttribute()
    {
        $city = Location::where('code', $this->city_id)->first()->name;
        $district = Location::where('code', $this->district_id)->first()->name;
        $ward = Location::where('code', $this->ward_id)->first()->name;
        return $this->address . ', ' . $ward . ', ' . $district . ', ' . $city;
    }

    public function getSumTotalAttribute()
    {
        return OrderDetail::where('order_id', $this->id)->get()->sum('total_price');
    }

    public static function getTextShip($str)
    {
        $str1 = '';
        if ($str == 'cod_hn') $str1 = OrderConstant::cod_hn;
        if ($str == 'cod_the_hn') $str1 = OrderConstant::cod_the_hn;
        if ($str == 'cod_sg') $str1 = OrderConstant::cod_sg;
        if ($str == 'cod_the_sg') $str1 = OrderConstant::cod_the_sg;
        if ($str == 'ship_hn') $str1 = OrderConstant::ship_hn;
        if ($str == 'shopee') $str1 = OrderConstant::shopee;
        if ($str == 'ship_sg') $str1 = OrderConstant::ship_sg;
        if ($str == 'tiki') $str1 = OrderConstant::tiki;
        if ($str == 'thue_tm') $str1 = OrderConstant::thue_tm;
        if ($str == 'thue_adam') $str1 = OrderConstant::thue_adam;
        if ($str == 'vnp') $str1 = OrderConstant::vnp;
        if ($str == 'ghn') $str1 = OrderConstant::ghn;
        if ($str == 'vnp_kh') $str1 = OrderConstant::vnp_kh;
        return $str1;
    }
}
