<?php

namespace App\Models;

use App\Constants\CustomerConstant;
use App\Constants\OrderConstant;
use App\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Helpers\Functions;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\DB;

class Customer extends Model
{
    protected $guarded = [];
    use SoftDeletes;

    public function source()
    {
        return $this->belongsTo(Source::class);
    }

    public function fanpagePost()
    {
        return $this->belongsTo(FanpagePost::class, 'post_id', 'post_id');
    }

    public function fanpage()
    {
        return $this->belongsTo(Fanpage::class, 'page_id', 'page_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function sale()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function wareHouse()
    {
        return $this->belongsTo(User::class, 'warehouse_id');
    }

    public function marketing()
    {
        return $this->belongsTo(User::class, 'mkt_id');
    }

    public function orders()
    {
        return $this->hasMany(Order::class, 'customer_id', 'id');
    }

    public function historyChangeStatus()
    {
        return $this->hasMany(HistoryChangeStatus::class);
    }

    public function getQuantityAttr($order)
    {
        $quantityText = $order->pluck('quantity')->toArray();
        return implode('</br> ', $quantityText);
    }

    public function getPriceAttr($order)
    {
        $quantityText = $order->pluck('price')->toArray();
        return implode('</br> ', $quantityText);
    }

    public function orderNew()
    {
        if (count($this->orders)) {
            $order = $this->orders->last();
            $this['order_status'] = $order->status;
            $array_service = OrderDetail::where('order_id', $order->id);
            $this['quantity_text'] = self::getQuantityAttr($array_service);
            $this['price_text'] = self::getPriceAttr($array_service);
            $this['sum_quantity'] = $array_service->sum('quantity');
            $this['sum_price'] = $array_service->sum('total_price');
            $array_service = $array_service->pluck('product_id')->toArray();
            $products = Product::whereIn('id', $array_service)->pluck('name')->toArray();
            $this['products'] = implode('</br> ', $products);
        }
    }

    public function getOrderTextAttribute()
    {
        if (count($this->orders)) {
            $order = $this->orders->last();
            $this['order_status'] = $order->status;
            $array_service = OrderDetail::where('order_id', $order->id);
            $this['quantity_text'] = self::getQuantityAttr($array_service);
            $this['price_text'] = self::getPriceAttr($array_service);
            $this['sum_quantity'] = $array_service->sum('quantity');
            $this['sum_price'] = $array_service->sum('total_price');
            $this['total_price'] = $order->total + $order->advance_price;
            $array_service = $array_service->pluck('product_id')->toArray();
            $products = Product::whereIn('id', $array_service)->pluck('name')->toArray();
            $this['products'] = implode('</br> ', $products);
        } else {

            if (isset($this->source) && $this->source) {
                $arrProduct = json_decode($this->source->product_id);
                $products = Product::whereIn('id', $arrProduct)->pluck('name')->toArray();
            } else {
                $products = [];
            }
            $this['products'] = implode('</br> ', $products);
        }
        return $this;
    }

    public function getOrderLast($orders)
    {
        $order = $orders->last();
        $result = OrderDetail::where('order_id', $order->id)->pluck('product_id')->toArray();
        return $result;
    }

    public function customerStatus()
    {
        return $this->belongsTo(CustomerStatus::class, 'status');
    }

    public function statusGroup()
    {
        return $this->belongsTo(CustomerStatus::class, 'status');
    }

    public static function search($input, $sort = 'updated_at')
    {
        $docs = self::when(isset($input['start_date']) && isset($input['end_date']),
            function ($q) use ($input, $sort) {
                if ($sort == 'ngay_chot_don') {
                    $q->whereHas('orders', function ($qw) use ($input) {
                        $qw->whereBetween('ngay_chot_don', [
                            Functions::yearMonthDayTime($input['start_date']),
                            Functions::yearMonthDayTime($input['end_date']),
                        ]);
                    });
                } elseif ($sort == 'ngay_tac_nghiep') {
                    $histoty_change = HistoryChangeStatus::select('customer_id')->groupBy('customer_id')
                        ->where('type', 0)->whereBetween('created_at', [
                            Functions::yearMonthDayTime($input['start_date']),
                            Functions::yearMonthDayTime($input['end_date']),
                        ])->pluck('customer_id')->toArray();
                    $q->whereIn('id', $histoty_change);
                } else {
                    $q->whereBetween($sort, [
                        Functions::yearMonthDayTime($input['start_date']),
                        Functions::yearMonthDayTime($input['end_date']),
                    ]);
                }
            })->when(isset($input['data_time']), function ($query) use ($input) {
            $query->when($input['data_time'] == 'TODAY' ||
                $input['data_time'] == 'YESTERDAY', function ($q) use ($input) {
                $q->whereDate('created_at', getTime(($input['data_time'])));
            })
                ->when($input['data_time'] == 'THIS_WEEK' ||
                    $input['data_time'] == 'LAST_WEEK' ||
                    $input['data_time'] == 'THIS_MONTH' ||
                    $input['data_time'] == 'LAST_MONTH', function ($q) use ($input) {
                    $q->whereBetween('created_at', getTime(($input['data_time'])));
                });
        })->when(isset($input['customer_not_position']), function ($q) use ($input) {
            $q->whereNotNull('user_id');
        })->when(isset($input['source_id']), function ($q) use ($input) {
            $q->whereIn('source_id', $input['source_id']);
        })->when(isset($input['customer_id']), function ($q) use ($input) {
            $q->where('id', $input['customer_id']);
        })->when(isset($input['sale_id']), function ($q) use ($input) {
            $q->where('user_id', $input['sale_id']);
        })->when(isset($input['is_cskh']), function ($q) use ($input) {
            $q->where('expired_time_boolean', 2);
        })->when(isset($input['arr_sale_id']), function ($q) use ($input) {
            $q->whereIn('user_id', $input['arr_sale_id']);
        })->when(isset($input['marketing']), function ($q) use ($input) {
            $q->where('mkt_id', $input['marketing']);
        })->when(isset($input['arr_mkt_id']), function ($q) use ($input) {
            $q->whereIn('mkt_id', $input['arr_mkt_id']);
        })->when(isset($input['search']), function ($q) use ($input) {
            $q->where('phone', 'like', '%' . $input['search']);
        })->when(isset($input['products']), function ($q) use ($input) {
            $q->whereHas("source", function ($s) use ($input) {
                $s->where('product_id', 'like', '%"' . $input['products'] . '"%');
            });
        })->when(isset($input['kieu_ngay']) && isset($input['start_date']) && isset($input['end_date']),
            function ($q) use ($input) {
                $q->whereHas("orders", function ($s) use ($input) {
                    $s->whereBetween('ngay_chot_don', [
                        Functions::yearMonthDayTime($input['start_date']),
                        Functions::yearMonthDayTime($input['end_date']),
                    ]);
                });
            })->when(isset($input['customer_status']), function ($q) use ($input) {
            $q->where('status', $input['customer_status']);
        })->when(isset($input['expired_time_boolean']), function ($q) use ($input) {
            $q->whereIn('expired_time_boolean', $input['expired_time_boolean']);
        })->when(isset($input['date_check_move']), function ($q) use ($input) {
            $q->where('time_move_sale', '<=', $input['date_check_move']);
        })->when(isset($input['date_check_expired']), function ($q) use ($input) {
            $q->where('expired_time', '<=', $input['date_check_expired']);
        })->when(isset($input['isset_check_time_expired']), function ($q) use ($input) {
            $q->whereNotNull('expired_time');
        })->when(isset($input['status_operation']), function ($q) use ($input) {
            if ($input['status_operation'] == 0) {
                $q->where('status', 0);
            } else {
                $q->where('status', '>', 0);
            }
        })->when(isset($input['chot_don']), function ($q) use ($input) {
            if ($input['chot_don'] == 0) {
                $order = Order::where('status',
                    0)->orderByDesc('id')->groupBy('customer_id')->pluck('customer_id')->toArray();
                $q->whereIn('id', $order);
            } else {
                $order = Order::where('status', '>', 0)->where('status', '<>',
                    OrderConstant::STATUS_HUY_DON)->orderByDesc('id')->groupBy('customer_id')->pluck('customer_id')->toArray();
                $q->whereIn('id', $order);
            }

        })
            ->when(isset($input['doi_soat']), function ($q) use ($input) {
                $q->whereHas('orders', function ($qw) use ($input) {
                    $qw->where('doi_soat', $input['doi_soat']);
                });
            })->when(isset($input['code']), function ($q) use ($input) {
                $q->whereHas('orders', function ($qw) use ($input) {
                    $qw->where('code', $input['code']);
                });
            })
            ->when(isset($input['order_status']), function ($q) use ($input) {
                $q->whereHas('orders', function ($qw) use ($input, $q) {
                    $qw->latest()->where('status', $input['order_status']);
                });
            })
            ->when(isset($input['giao_hang']), function ($q) use ($input) {
                $arr_orders = Order::select('customer_id', DB::raw('MAX(id) as max_id'))
                    ->whereBetween('ngay_chot_don', [
                        Functions::yearMonthDayTime($input['start_date']),
                        Functions::yearMonthDayTime($input['end_date']),
                    ])
                    ->groupBy('customer_id')->pluck('max_id')->toArray();

                $order = Order::select('customer_id')->whereIn('id', $arr_orders)->where('status', $input['giao_hang'])
                    ->pluck('customer_id');
                $q->whereIn('id', $order);

            })->when(isset($input['customer_old']), function ($q) use ($input) {
                $q->where('customer_old', $input['customer_old']);
            })->when(isset($input['gioi_thieu']), function ($q) use ($input) {
                $q->where('gioi_thieu', $input['gioi_thieu']);
            })->when(isset($input['duplicate']), function ($q) use ($input) {
                $q->where('duplicate', $input['duplicate']);
            })->orderByDesc('id');
        return $docs;
    }

    public static function search1($input, $sort = 'updated_at')
    {
        $docs = self::when(isset($input['start_date']) && isset($input['end_date']), function ($q) use ($input, $sort) {
            $q->whereBetween($sort, [
                Functions::yearMonthDayTime($input['start_date']),
                Functions::yearMonthDayTime($input['end_date']),
            ]);
        })->when(isset($input['giao_hang']), function ($q) use ($input) {
            $order = Order::where('status',
                $input['giao_hang'])->orderByDesc('id')->groupBy('customer_id')->pluck('customer_id')->toArray();
            $q->whereIn('id', $order);
        })
            ->when(isset($input['source_id']), function ($q) use ($input) {
                $q->where('source_id', $input['source_id']);
            })
            ->when(isset($input['arr_sale_id']), function ($q) use ($input) {
                $q->whereIn('user_id', $input['arr_sale_id']);
            })
            ->when(isset($input['arr_mkt_id']), function ($q) use ($input) {
                $q->whereIn('mkt_id', $input['arr_mkt_id']);
            })
            ->when(isset($input['in_status']), function ($q) use ($input) {
                $q->whereIn('status', $input['in_status']);
            })
            ->when(isset($input['marketing']), function ($q) use ($input) {
                $q->where('mkt_id', $input['marketing']);
            })
            ->when(isset($input['customer_old']), function ($q) use ($input) {
                $q->where('customer_old', $input['customer_old']);
            })->when(isset($input['gioi_thieu']), function ($q) use ($input) {
                $q->where('gioi_thieu', $input['gioi_thieu']);
            });
        return $docs;
    }

    public function getProductTextAttribute()
    {
        if (count($this->orders)) {
            $order = $this->orders->last();
            $array_service = OrderDetail::where('order_id', $order->id);
            $array_service = $array_service->pluck('product_id')->toArray();
            $products = Product::whereIn('id', $array_service)->pluck('name')->toArray();
        } else {
            if (isset($this->source) && $this->source) {
                $arrProduct = json_decode($this->source->product_id);
                $products = Product::whereIn('id', $arrProduct)->pluck('name')->toArray();
            } else {
                $products = [];
            }


        }
        return implode('</br> ', $products);
    }

    /**
     * get trạng thái con và trạng thái hiện tại
     *
     * @return array
     */
    public function getCustomerStatusAttribute()
    {
        $arr = [];
        $child = CustomerStatus::find($this->status);
        if (empty($child)) {
            $arr = CustomerStatus::all();
        } else {
            array_push($arr, $child->id);
            if ($child->customer_child) {
                $arr_child = json_decode($child->customer_child);
                foreach ($arr_child as $item) {
                    $arr[] = $item;
                }
            }
            $arr = CustomerStatus::whereIn('id', $arr)->get();
        }
        return $arr;
    }

    /**
     * get lịch sử note cuối cùng
     *
     * @return string
     */
    public function getHistoryNoteAttribute()
    {
        $sale_note = HistoryChangeStatus::where('customer_id', $this->id)->get();
        if (count($sale_note)) {
            $sale_note = $sale_note->last()->note;
            $sale_note = strip_tags($sale_note);
        } else {
            $sale_note = '';
        }
        return $sale_note;
    }

    /**
     * get thời gian quá hạn để update
     *
     * @param $status_id
     *
     * @return int
     */
    public static function timeExpired($status_id)
    {
        $date = date('Y-m-d H:i:s');
        $customer_stauts = CustomerStatus::find($status_id);
        $time_expired = $customer_stauts->expired_time;
        $time_move = $customer_stauts->time_move;

        $data['expired_time'] = empty($time_expired) ? null : date('Y-m-d H:i:s',
            strtotime('+' . $time_expired . 'minute', strtotime($date)));
        $data['time_move_sale'] = empty($time_move) ? null : date('Y-m-d H:i:s',
            strtotime('+' . $time_move . 'minute', strtotime($data['expired_time'])));

        return $data;
    }

    public function getExpiredTextAttribute()
    {
        $now = Date::now()->format('Y-m-d H:i:s');

        if (!empty($this->expired_time)) {
            $countdown = strtotime($this->expired_time) - strtotime($now);
            $days = ($countdown / 86400) >= 1 ? floor($countdown / 86400) : 0;
            $hours = floor(($countdown % 86400) / 3600);
            $minutes = round((($countdown % 86400) % 3600) / 60);
            return ($days > 0 ? $days . ' ngày &nbsp;' : '') . ($hours > 0 ? $hours . ' giờ &nbsp;' : '') . ($minutes > 0 && $days < 1 ? $minutes . ' phút' : '');
        }
        return '';
    }

    public function getTimeMoveAttribute()
    {
        $now = Date::now()->format('Y-m-d H:i:s');

        if (!empty($this->time_move_sale)) {
            $countdown = strtotime($this->time_move_sale) - strtotime($now);
            $days = ($countdown / 86400) >= 1 ? floor($countdown / 86400) : 0;
            $hours = floor(($countdown % 86400) / 3600);
            $minutes = round((($countdown % 86400) % 3600) / 60);
            return ($days > 0 ? $days . ' ngày &nbsp;' : '') . ($hours > 0 ? $hours . ' giờ &nbsp;' : '') . ($minutes > 0 && $days < 1 ? $minutes . ' phút' : '');
        }
        return '';
    }

    public function getLinkMoveAttribute()
    {
        if (!empty($this->post_id)) {
            return 'https://fb.com/' . @$this->fanpagePost->page_id . '/posts/' . $this->post_id;
        } else {
            return $this->source->url_source;
        }
    }

    public function latestOfMany($column = 'id', $relation = null)
    {
        return $this->ofMany(collect(Arr::wrap($column))->mapWithKeys(function ($column) {
            return [$column => 'MAX'];
        })->all(), 'MAX', $relation ?: $this->guessRelationship());
    }

//    public function lastOrder()
//    {
//        return $this->hasOne(Order::class, 'customer_id','id')->ofMany(collect(Arr::wrap($column))->mapWithKeys(function ($column) {
//            return [$column => 'MAX'];
//        })->all(), 'MAX', $relation ?: $this->guessRelationship());;
//    }
}
